/*
 * This file is part of system-settings
 *
 * Copyright (C) 2017 The UBports project
 * Copyright (C) 2020 UBports Foundation
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Marius Gripsgard <marius@ubports.com>
 */

import QtQuick 2.4
import SystemSettings 1.0
import Lomiri.Components 1.3
import Lomiri.Components.ListItems 1.3 as ListItem
import Lomiri.Connectivity 1.0
import Lomiri.SystemSettings.Update 1.0
import "i18nd.js" as I18nd

ItemPage {
    id: root
    objectName: "channelSettingsPage"
    title: I18nd.tr("Channel settings")

    signal channelChanged()
    property bool isSwitching: false

    Column {
        id: aIndicator
        anchors.horizontalCenter: parent.horizontalCenter
        anchors.verticalCenter: parent.verticalCenter
        visible: !channelSelectorModel.isLoaded || root.isSwitching

        ActivityIndicator {
            opacity: visible ? 1 : 0
            visible: parent.visible
            running: visible
            anchors.horizontalCenter: parent.horizontalCenter
        }

        Label {
            id: aIndicatorText
            text: I18nd.tr("Fetching channels")
            visible: parent.visible
        }
    }

    ListItem.ItemSelector {
        id: channelSelector
        anchors.top: root.header.bottom
        expanded: true
        text: I18nd.tr ("Channel to get updates from:")
        model: channelSelectorModel
        delegate: selectorDelegate
        visible: !aIndicator.visible

        onDelegateClicked: {
            aIndicatorText.text = I18nd.tr("Switching channel");

            const channel = channelSelectorModel.get(index).channel
            root.isSwitching = true;
            ChUtils.switchChannel(channel).then(function() {
              root.isSwitching = false;
              root.channelChanged()
            });
        }
    }

    Component {
        id: selectorDelegate
        OptionSelectorDelegate { text: name; subText: description; }
    }

    ListModel {
        id: channelSelectorModel
        property bool isLoaded: false

        Component.onCompleted: {
          ChUtils.getAvailableChannels().then(function (channels) {
            let currentChannel = SystemImage.getSwitchChannel();
            let currentChInfo = ChUtils.parseChannel(currentChannel);

            let i = 0;
            for (let [chName, chInfo] of Object.entries(channels)) {
              if (chInfo.hidden)
                continue;
              if (chInfo.series != currentChInfo.series) {
                // Series upgrade is handled elsewhere.
                continue;
              }

              channelSelectorModel.append({
                name: chInfo.prettyName,
                description: "",
                channel: chName
              })

              if (chName == currentChannel)
                channelSelector.selectedIndex = i;

              i++;
            }

            channelSelectorModel.isLoaded = true;
          })
        }
    }
}
