/*
 * Copyright (C) 2025 UBports Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 * Alfred Neumayer <alfred@ubports.com>
 *
 */

import QtQuick 2.4
import QtQuick.Layouts 1.12
import SystemSettings 1.0
import SystemSettings.ListItems 1.0 as SettingsListItems
import Lomiri.Components 1.3
import Lomiri.Connectivity 1.0
import Lomiri.Components.ListItems 1.3 as ListItems
import Lomiri.Components.Popups 1.3
import Lomiri.SystemSettings.SecurityPrivacy 1.0

ItemPage {
    id: encryptionPage
    title: i18n.tr("Encryption (Experimental)")
    flickable: scrollWidget

    LomiriSecurityPrivacyPanel {
        id: securityPrivacy
    }

    readonly property bool supported : securityPrivacy.supportsFileSystemEncryption()
    readonly property bool isSetUp : securityPrivacy.encryptionIsSetUp()
    readonly property bool usesV1Policies : securityPrivacy.usesV1EncryptionPolicies()

    // Something disallows Q_ENUM + qRegisterMetaType to work, so maintain a mapping ourselves...
    readonly property int reason_unknown : 0
    readonly property int reason_setup_failed : 1
    readonly property int reason_storage_low_failed: 2
    readonly property int reason_encryption_failed : 3
    readonly property int reason_migration_failed : 4
    readonly property int reason_disable_autologin_failed : 5
    readonly property int reason_cleanup_failed : 6
    readonly property int reason_ending_failed : 7

    function reasonToErrorString(reason) {
        switch (reason) {
        case reason_setup_failed:
            return i18n.tr("Setup failed, please try again later.");
        case reason_storage_low_failed:
            return i18n.tr("Storage is low. Please make available space by removing unnecessary files and try again.")
        case reason_encryption_failed:
            return i18n.tr("Encryption failed. Please try again later.");
        case reason_migration_failed:
            return i18n.tr("Migration failed. Please make available space by removing old files and try again.")
        case reason_disable_autologin_failed:
            return i18n.tr("Disabling autologin failed. Please report this incident to the developers.");
        case reason_cleanup_failed:
            return i18n.tr("Cleanup failed. This might indicate that migration has succeeded. Please reboot the device.")
        case reason_ending_failed:
            return i18n.tr("Ending phase failed. Please reboot the device manually.")
        case reason_unknown:
        default:
            return i18n.tr("Unknown error");
        }
    }

    Component {
        id: errorDialog
        Dialog {
            id: errorDialogue
            property string errorMessage : ""

            title: i18n.tr("Error")
            text: errorMessage

            Button {
                text: "Cancel"
                onClicked: {
                    PopupUtils.close(errorDialogue)
                }
            }
        }
    }

    Component {
        id: pinDialog
        Dialog {
            id: pinDialogue
            title: i18n.tr("Encryption")
            text: working ? i18n.tr("Encrypting device, please don't close this window.") :
                            i18n.tr("Please enter your PIN or password to continue")

            property bool working : false

            Connections {
                target: securityPrivacy
                function onEncryptionErrorOccurred(reason) {
                    console.log("Error occurred: " + reason)
                    pinDialogue.working = false;
                    PopupUtils.close(pinDialogue)
                    const errorString = reasonToErrorString(reason)
                    PopupUtils.open(errorDialog, encryptionPage, { 'errorMessage' : errorString})
                }
            }

            SequentialAnimation {
                id: passwordWrongAnimation
                NumberAnimation { target: password; property: "x"; to: -units.gu(2); duration: 100 }
                NumberAnimation { target: password; property: "x"; to: units.gu(2); duration: 100 }
                NumberAnimation { target: password; property: "x"; to: 0; duration: 100 }
            }

            TextField {
                id: password
                placeholderText: i18n.tr("PIN or password")
                echoMode: TextInput.Password
                focus: true
                visible: !working
            }
            Button {
                text: "OK"
                visible: !working
                color: theme.palette.normal.positive
                onClicked: {
                    if (!securityPrivacy.verifyUserPassword(password.text)) {
                        passwordWrongAnimation.restart()
                        return;
                    }

                    working = true;
                    securityPrivacy.startEncryption(password.text);
                }
            }
            Button {
                text: "Cancel"
                visible: !working
                onClicked: {
                    PopupUtils.close(pinDialogue)
                }
            }

            ActivityIndicator {
                id: activity
                visible: working
                running: working
                anchors.centerIn: parent
            }
        }
    }

    Flickable {
        id: scrollWidget
        anchors.fill: parent
        contentHeight: contentItem.childrenRect.height
        boundsBehavior: (contentHeight > encryptionPage.height) ? Flickable.DragAndOvershootBounds : Flickable.StopAtBounds
        /* Set the direction to workaround https://bugreports.qt-project.org/browse/QTBUG-31905
           otherwise the UI might end up in a situation where scrolling doesn't work */
        flickableDirection: Flickable.VerticalFlick

        ColumnLayout {
            anchors.left: parent.left
            anchors.right: parent.right

            Icon {
                width: parent.width / 3
                height: width
                Layout.alignment: Qt.AlignCenter
                Layout.topMargin: units.gu(4)
                name: "preferences-system-privacy-symbolic"
            }

            SettingsItemTitle {
                text: i18n.tr("About encryption on your device:")
            }

            ListItems.Caption {
                text: i18n.tr(
                          "• Encryption protects against access to phone data when the phone is connected to a PC or other device.") + "\n\n"
                visible: encryptionPage.supported
                Layout.preferredWidth: parent.width
            }

            ListItems.Caption {
                text: i18n.tr(
                          "• This device generation doesn't support encryption and container technologies with user data access at the same time. " +
                          "Beware that this will break technologies like Waydroid and Docker in certain ways.") + "\n\n"
                visible: encryptionPage.usesV1Policies
                Layout.preferredWidth: parent.width
            }

            ListItems.Caption {
                text: i18n.tr(
                          "• Data will be migrated to encrypted storage. The device needs some free space for this migration. " +
                          "Please make room by deleting files and backing them up to another place. " +
                          "Ensure that you have enough free space available before proceeding with the encryption process.") + "\n\n"
                visible: !encryptionPage.isSetUp
                Layout.preferredWidth: parent.width
            }

            Button {
                text: i18n.tr("Encrypt device")
                color: LomiriColors.green
                visible: !encryptionPage.isSetUp
                Layout.alignment: Qt.AlignCenter

                onClicked: {
                    PopupUtils.open(pinDialog)
                }
            }

            Label {
                font.bold: true
                visible: encryptionPage.isSetUp
                text: i18n.tr("Encryption is enabled.")
                Layout.alignment: Qt.AlignCenter
                Layout.bottomMargin: units.gu(4)
            }
        }
    }
}
